(*-------------------------------------------------------------------------------------------*)
(* 1 : TYPES *)
(*-------------------------------------------------------------------------------------------*)

type mouvement1 =
	{mutable mv1: (int vect * int vect vect) list}
;;

type context =
	{mutable matrice: int vect vect}
;;

type repere = {mutable plan: int * int * int * int};;

type ops = OPS of (unit -> unit) * (unit -> unit) * (unit -> unit);;

type couleur = ORANGE | VERT | BLANC | ROUGE | BLEU | JAUNE | GRIS;;

type bouton = {titre: string;
		orx: int; ory: int;
		largeur: int; hauteur: int;
		mutable couleur: couleur;
		mutable action: unit -> unit;
		mutable bas: bool};;

type cube1 =
	{
		anime1: bool ref;
		mutable mouvement1: mouvement1;
		mutable mvi: mouvement1;
		mutable context1: context;
		mutable repere1: repere;
		mutable rotations_cube1: ops * ops;
		mutable rotations_faces1: ops * ops * ops * ops;
		mutable boutons1: bouton vect;
	}
;;

type couple = COUPLE of (int vect * int vect) | NIL;;

(*-------------------------------------------------------------------------------------------*)
(* 2 : DIVERS *)
(*-------------------------------------------------------------------------------------------*)

(* matrices en dimension 3 *)
let matrice_nulle = [|[|0; 0; 0|]; [|0; 0; 0|]; [|0; 0; 0|]|];;

let vect v = if vect_length v = 3 then (v.(0), v.(1), v.(2))
	else failwith "vect"
;;

let matscal a = let m = make_matrix 3 3 0 in
		for i = 0 to 2 do
			m.(i).(i) <- a
		done;
		m
;;

let id = matscal 1 and idm = matscal (- 1);;

(* produit du vecteur ligne entier v par la matrice entière m *)
let prefix /:/ v m =
	let w j = let t = ref 0 in for k = 0 to vect_length v - 1 do
				t := !t + m.(k).(j) * v.(k) done;
			!t in
		[|w 0; w 1; w 2|]
;;

(*produit du scalaire a par la matrice m*)
let prefix /../ a m =
	map_vect (fun x -> map_vect (fun t -> a * t) x) m;;

(* produit matriciel *)
let prefix /./ m m1 = map_vect (fun v -> v /:/ m1) m;;

(* somme matricielle *)
let prefix /+/ m1 m2 =
	let m = make_matrix 3 3 0 in
		for i = 0 to 2 do
			for j = 0 to 2 do
				m.(i).(j) <- m1.(i).(j) + m2.(i).(j)
			done;
		done;
		m
;;

(* matrice diagonale *)
let diag a b c = [|[|a; 0; 0|]; [|0; b; 0|]; [|0; 0; c|]|];;

(* transposée de la matrice m  qui en est aussi l'inverse : *)
(* quand m est orthogonale *)
let transpose m =
	let m1 = make_matrix 3 3 0 in
		for i = 0 to 2 do
			for j = 0 to 2 do
				m1.(j).(i) <- m.(i).(j)
			done;
		done;
		m1
;;

(* produit scalaire *)
let prefix /|/ v w = v.(0) * w.(0) + v.(1) * w.(1) + v.(2) * w.(2);;

(* matrices des rotations d'un quart de tour autour des axes : *)
(* (opèrent à droite sur les lignes) *)

(* sens des aiguilles d'une montre *)
let rot v = match list_of_vect v with
		| [1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; - 1|]; [|0; 1; 0|]|]
		| [0; 1; 0] -> [|[|0; 0; 1|]; [|0; 1; 0|]; [|- 1; 0; 0|]|]
		| [0; 0; 1] -> [|[|0; - 1; 0|]; [|1; 0; 0|]; [|0; 0; 1|]|]
		| [- 1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; 1|]; [|0; - 1; 0|]|]
		| [0; - 1; 0] -> [|[|0; 0; - 1|]; [|0; 1; 0|]; [|1; 0; 0|]|]
		| [0; 0; - 1] -> [|[|0; 1; 0|]; [|- 1; 0; 0|]; [|0; 0; 1|]|]
		| _ -> failwith "rot"
;;

(* sens inverse des aiguilles d'une montre *)
let rot' v = match list_of_vect v with
		| [1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; 1|]; [|0; - 1; 0|]|]
		| [0; 1; 0] -> [|[|0; 0; - 1|]; [|0; 1; 0|]; [|1; 0; 0|]|]
		| [0; 0; 1] -> [|[|0; 1; 0|]; [|- 1; 0; 0|]; [|0; 0; 1|]|]
		| [- 1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; - 1|]; [|0; 1; 0|]|]
		| [0; - 1; 0] -> [|[|0; 0; 1|]; [|0; 1; 0|]; [|- 1; 0; 0|]|]
		| [0; 0; - 1] -> [|[|0; - 1; 0|]; [|1; 0; 0|]; [|0; 0; 1|]|]
		| _ -> failwith "rot'"
;;

(* liste dans l'ordre des éléments de l satisfaisant 'critère' *)
let rec select critere l = match l with
		t :: r -> let l1 = select critere r in if critere t then t :: l1 else l1
		| _ -> []
;;

(* liste des entiers de 0 à n - 1 *)
let liste n =
	let v = make_vect n 0 in
		for i = 0 to n - 1 do
			v.(i) <- i
		done;
		list_of_vect v
;;

(* permutation aléatoire des éléments d'une liste l *)
let random_list l =
	let n = list_length l and l1 = ref []
	in
		for i = 0 to n - 1 do
			l1 := (vect_of_list (subtract l !l1)).(random__int (n - i)) :: !l1
		done;
		!l1
;;

(* signature de la permutation p des éléments de la liste l *)
let sign l p =
	let n = list_length l and v = vect_of_list l
	and m = ref 1 in
		for i = 0 to n - 1 do
			for j = i + 1 to n - 1 do
				let a = v.(i) and b = v.(j) in
					if p a > p b && b > a || p b > p a && a > b then m := - !m;
			done;
		done;
		!m
;;

(* exécution d'une liste de mouvements *)
let rec exe1 l = match l with
		t :: r -> t (); exe1 r;
		| [] -> ()
;;

(* liste des noms des opérations du cube inscrits dans la chaîne s *)
let tete s =
	let l = string_length s in
		if l = 0 then ""
		else if l = 1 then sub_string s 0 1
		else if l = 2 then if s.[1] = `0` || s.[1] = `'` then s else sub_string s 0 1
		else match s.[1], s.[2] with
				| `0`, `'` -> sub_string s 0 3
				| `0`, _ -> sub_string s 0 2
				| `'`, _ -> sub_string s 0 2
				| _ -> sub_string s 0 1
;;
let scinde s =
	let t = tete s and ls = string_length s in
		let lt = string_length t in
			let r = sub_string s lt (ls - lt) in
				(t, r)
;;
let rec op_names_from_string s =
	let (t, r) = scinde s in
		if r = "" then [t] else t :: op_names_from_string r
;;

(* pour sortie vers caml *)
let format_string op_names_string = "exec \"" ^ op_names_string ^ "\";;\n";;

(*-------------------------------------------------------------------------------------------*)
(* 3 : INDICES *)
(*-------------------------------------------------------------------------------------------*)

(* indices des angles et des coins et des centres des faces *)
let indices = let l = ref [] in
		for k = 1 downto - 1 do
			for j = 1 downto - 1 do
				for i = 1 downto - 1 do l := [|i; j; k|] :: !l
				done
			done
		done;
		subtract !l [[|0; 0; 0|]]
;;

let est_centre x = x /|/ x = 1;;
let est_angle x = x /|/ x = 2;;
let est_coin x = x /|/ x = 3;;

(* liste des centres *)
let centres = select est_centre indices;;

(* liste des angles *)
let angles = select est_angle indices;;

(* liste des coins *)
let coins = select est_coin indices;;


(*-------------------------------------------------------------------------------------------*)
(* 4 : GROUPE DU CUBE *)
(*-------------------------------------------------------------------------------------------*)

let groupe_du_cube =
	[
		[|[|1; 0; 0|]; [|0; 1; 0|]; [|0; 0; 1|]|];
		[|[|1; 0; 0|]; [|0; - 1; 0|]; [|0; 0; - 1|]|];
		[|[|- 1; 0; 0|]; [|0; 1; 0|]; [|0; 0; - 1|]|];
		[|[|- 1; 0; 0|]; [|0; - 1; 0|]; [|0; 0; 1|]|];
		[|[|1; 0; 0|]; [|0; 0; - 1|]; [|0; 1; 0|]|];
		[|[|0; 0; 1|]; [|0; 1; 0|]; [|- 1; 0; 0|]|];
		[|[|0; - 1; 0|]; [|1; 0; 0|]; [|0; 0; 1|]|];
		[|[|1; 0; 0|]; [|0; 0; 1|]; [|0; - 1; 0|]|];
		[|[|0; 0; - 1|]; [|0; 1; 0|]; [|1; 0; 0|]|];
		[|[|0; 1; 0|]; [|- 1; 0; 0|]; [|0; 0; 1|]|];
		[|[|0; - 1; 0|]; [|- 1; 0; 0|]; [|0; 0; - 1|]|];
		[|[|0; 1; 0|]; [|1; 0; 0|]; [|0; 0; - 1|]|];
		[|[|- 1; 0; 0|]; [|0; 0; - 1|]; [|0; - 1; 0|]|];
		[|[|0; 0; - 1|]; [|0; - 1; 0|]; [|- 1; 0; 0|]|];
		[|[|0; 0; 1|]; [|0; - 1; 0|]; [|1; 0; 0|]|];
		[|[|- 1; 0; 0|]; [|0; 0; 1|]; [|0; 1; 0|]|];
		[|[|0; 1; 0|]; [|0; 0; 1|]; [|1; 0; 0|]|];
		[|[|0; 0; - 1|]; [|- 1; 0; 0|]; [|0; 1; 0|]|];
		[|[|0; 0; 1|]; [|- 1; 0; 0|]; [|0; - 1; 0|]|];
		[|[|0; 1; 0|]; [|0; 0; - 1|]; [|- 1; 0; 0|]|];
		[|[|0; 0; - 1|]; [|1; 0; 0|]; [|0; - 1; 0|]|];
		[|[|0; - 1; 0|]; [|0; 0; - 1|]; [|1; 0; 0|]|];
		[|[|0; - 1; 0|]; [|0; 0; 1|]; [|- 1; 0; 0|]|];
		[|[|0; 0; 1|]; [|1; 0; 0|]; [|0; 1; 0|]|]
	]
;;

(* représentation des (inverses des) éléments du groupe par les éléments a0, d0 ,h0 *)
let decomposition r =
	let lops = [""; "a0a0"; "d0d0"; "h0h0"; "a0'"; "d0'"; "h0'"; "a0"; "d0"; "h0";
			"h0'a0a0"; "h0a0a0"; "d0d0a0"; "d0a0a0"; "d0'a0a0"; "d0d0a0'"; "h0a0";
			"d0a0'"; "d0'a0"; "h0a0'"; "d0a0"; "h0'a0'"; "h0'a0"; "d0'a0'"]
	in assoc r (map2 (fun x y -> (x, y)) groupe_du_cube lops)
;;


(*-------------------------------------------------------------------------------------------*)
(* 5 : GROUPE DES MOUVEMENTS *)
(*-------------------------------------------------------------------------------------------*)

(* groupe M des mouvements des minicubes *)

(* tri d'un mouvement selon l'ordre des indices *)
let trier mv1 = sort__sort (fun x y -> fst x < fst y) mv1;;

(* élément neutre de M *)
let e = map (fun x -> x, id) indices;;

(* conversion entre mouvement représenté par une fonction et mouvement *)
(* représenté par une liste : (int vect * int vect vect) list *)
let mv1_of_fun f =
	map (fun (x, y) -> (x, y /./ (f x))) e
;;
let fun_of_mv1 mv1 x =
	assoc x mv1
;;

(* mouvements globaux *)
let cst x = mv1_of_fun (fun t -> x);;

(* loi interne *)
let prefix /*/ mv1 mv1' =
	let f = fun_of_mv1 mv1 and f' = fun_of_mv1 mv1'
	in
		let s t = t /:/ (f t)
		in trier (mv1_of_fun (fun x -> (f x) /./ (f' (s x))))
;;

(* inverse d'un élément *)
let inverse mv1 = map (fun (x, y) -> (x /:/ y, transpose y)) mv1;;

(* mouvements de Rubik élémentaires *)

(* rotations dans le sens des aiguilles d'une montre d'un quart de tour de la *)
(* face - tranche interne dans le cas du cube 4x4 - normale au vecteur sortant 'v' *)
let rub v = mv1_of_fun
	(fun x -> if (x /|/ v) = 1 then rot v else id)
;;

(* mouvement inverse du précédent *)
let rub' v = inverse (rub v);;

(* enregistrement sur disque d'un mouvement: format portable *)
let enregistrer_mouv mv chemin =
	let rec aux mv =
		let traite x =
			printf__sprintf "%d%d%d" x.(0) x.(1) x.(2)
		in
			match mv with
				| [] -> ""
				| t :: r ->
							let (x, m) = t
							in
								traite x ^ traite m.(0) ^ traite m.(1) ^ traite m.(2);
								^ aux r
	in
		try
			let canalout = open_out chemin
			in
				output_string canalout (aux mv);
				close_out canalout
		with sys__Sys_error s -> failwith s
;;

(* lecture sur disque d'un mouvement : format portable *)
let couple_of_int_matrice s =
	let t = make_matrix 4 3 0 in
		for i = 0 to 3 do
			for j = 0 to 2 do
				t.(i).(j) <- s.(i * 3 + j)
			done
		done;
		(t.(0), [|t.(1); t.(2); t.(3)|])
;;
let int_vect s =
	let tete s =
		let l = string_length s in
			if l = 0 then ""
			else if s.[0] = `-` then sub_string s 0 2
			else sub_string s 0 1
	in
		let reste s =
			let l = string_length s
			and lt = string_length (tete s) in
				sub_string s lt (l - lt)
		in
			if s = "" then [||]
			else
				let rec aux ss =
					let t = tete ss and r = reste ss in
						if r <> "" then t :: aux r
						else [t]
				in vect_of_list (map int_of_string (aux s))
;;
let int_matrices_of_int_vect v =
	let lst = ref [] in
		for i = 0 to (vect_length v - 12) / 12 do
			lst := sub_vect v (12 * i) 12 :: !lst
		done;
		vect_of_list !lst
;;
let lire_mouv path =
	try
		let canalin = open_in path in
			let s = input_line canalin in
				close_in canalin;
				rev (list_of_vect (map_vect couple_of_int_matrice (int_matrices_of_int_vect (int_vect s))))
	with sys__Sys_error s1 -> print_string s1; e
;;

(*-------------------------------------------------------------------------------------------*)
(* 6 : TEST DE VALIDITÉ D'UN MOUVEMENT (APPARTENANCE AU SOUS-GROUPE DE RUBIK *)
(*-------------------------------------------------------------------------------------------*)

let marque x =
	if est_coin x then [|0; 0; x.(2)|]
	else if est_angle x then let a, b, c = x.(0), x.(1), x.(2)
		in match a, b, c with
				| 0, _, _ -> [|0; b; 0|]
				| _, 0, _ -> [|0; 0; c|]
				| _, _, 0 -> [|a; 0; 0|]
				| _ -> [|0; 0; 0|]
	else [|0; 0; 0|]
;;

(* morphisme 's: M -> S' et section 'l: S -> M' *)
(* construction d'une section 'l' de la suite exacte '0 -> K -> M -> S -> 0' *)
(* En Caml on représente la sujection 's' par 'sur', la section 'l' par 'sec' et 'gij' par 'gg i j' *)

(* éléments g_{ij} alias gg i j de G servant à construire cette section *)
let gg i j =
	let critere i j g = i /:/ g = j && marque i /:/ g = marque j
	in
		hd (select (critere i j) groupe_du_cube)
		(* cette liste devrait toujours contenir exactement un élément *)
;;

(* décomposition 'm = ker m /*/ sec (sur m)' d'un mouvement 'm' *)
(* avec 'ker m' élément du noyau de 'sur' *)
(* 'p' pour 'permutation': 'p = sur m' est la permutation 'p' des indices associée au mouvement 'm' *)
let sec p = mv1_of_fun (fun i -> gg i (p i));;
let sur m = fun i -> i /:/ fun_of_mv1 m i;;
let ker m = m /*/ inverse (sec (sur m));;

(* stabilisateurs des angles et des coins *)
let st i =
	let stc i =
		let x = [|1; 1; 1|]
		and m = [|[|0; 0; 1|]; [|1; 0; 0|]; [|0; 1; 0|]|]
		in
			gg i x /./ m /./ gg x i
	and sta i = let x = [|1; 0; 1|]
		and m = [|[|0; 0; 1|]; [|0; - 1; 0|]; [|1; 0; 0|]|]
		in
			gg i x /./ m /./ gg x i
	in
		if est_angle i then sta i else if est_coin i then stc i else failwith "st"
;;

(* rotation totale des angles *)
let rta mv1 =
	let rta_aux k = let f = fun_of_mv1 k in
			let indexa i = if f i = st i then 1 else 0 in
				(list_it (prefix +) (map indexa angles) 0) mod 2
	in rta_aux (ker mv1)
;;

(* rotation totale des coins *)
let rtc mv1 =
	let rtc_aux k = let f = fun_of_mv1 k in
			let indexc i = if f i = st i then 1
				else if f i = transpose (st i) then 2 else 0 in
				(list_it (prefix +) (map indexc coins) 0) mod 3
	in rtc_aux (ker mv1)
;;
(* test d'appartenance d'un mouvement au sous-groupe de Rubik R *)
(* par nullité des rotations totales et égalité des signatures *)
(* des permutations d'angles et de coins *)
let est_dans_R mv1 = let p = sur mv1 in
		sign angles p = sign coins p && rta mv1 = 0 && rtc mv1 = 0
;;

(* mouvement général de type 'mv1' défini par des permutations et rotations d'angles et de coins et des exposants d'angles et de coins *)
(* avec prise en compte de la rotation des milieux des faces *)
let nouveau_mv1 pa pc ea ec em =
	let k = mv1_of_fun
		(fun i ->
							if est_angle i then if ea i = 0 then id else st i
							else if est_coin i then
								let e = ec i in
									if e = 0 then id
									else if e = 1 then st i
									else transpose (st i)
							else if est_centre i then
								let e = em i in
									if e = 0 then id else if e = 1 then rot i else if e = 2 then rot i /./ rot i else rot' i
							else id
		)
	and l = mv1_of_fun
		(
			fun i ->
							if est_angle i then gg i (pa i) else if est_coin i then gg i (pc i)
							else id
		)
	in k /*/ l
;;

(* prise en compte de la rotation des minicubes centraux des faces *)
(* nombre de quarts de tour du centre de la face orthogonale au vecteur sortant 'v' *)
(* nombre de quarts de tour de la face normale au vecteur 'v' dans le mouvement 'mv1' *)
let nqt mv1 v =
	let r = fun_of_mv1 mv1 v in
		if r = id then 0
		else if r = rot v then 1
		else if r = rot v /./ rot v then 2
		else if r = rot' v then 3
		else failwith "nqt"
;;

(* nombre total de quarts de tour des faces dans le mouvement mv1 *)
let nqt_total mv1 = (list_it (prefix +) (map (nqt mv1) (list_of_vect id @ list_of_vect idm)) 0) mod 4;;

(* rotation horaire d'un quart de tour du centre orthogonal à v *)
let ajuste_centre mv1 r v = mv1_of_fun (fun x -> if x = v then (fun_of_mv1 mv1 v) /./ r v else fun_of_mv1 mv1 x);;

(*- mouvement aléatoire -*)

(* permutation aléatoire d'une liste *)
let pl_r l =
	let l' = random_list l
	in fun i -> assoc i (map2 (fun x y -> x, y) l l')
;;

(* exposant aléatoire pour les angles *)
let ea_r i = if est_angle i then random__int 2 else failwith "ea_r";;

(* exposant aléatoire pour les coins *)
let ec_r i = if est_coin i then random__int 3 else failwith "ec_r";;

(* exposant aléatoire pour les centres *)
let em_r i = if est_centre i then random__int 3 else failwith "em_r";;

(* mouvement aléatoire général *)
let mv1_r () = nouveau_mv1 (pl_r angles) (pl_r coins) ea_r ec_r em_r;;

let mv1_rubik_r () =
	(* rotation d'un seul angle i *)
	let rot_angle i n =
		nouveau_mv1 (fun x -> x) (fun x -> x) (fun x -> if x = i then n else 0) (fun x -> 0) (fun x -> 0)
	and
	(* rotation d'un seul coin i *)
	rot_coin i n =
		nouveau_mv1 (fun x -> x) (fun x -> x) (fun j -> 0) (fun j -> if j = i then n else 0) (fun x -> 0)
	in
		let m = ref (mv1_r ()) in
			if rta !m <> 0 then m := !m /*/ rot_angle [|0; 1; 1|] 1;
			if rtc !m <> 0 then m := !m /*/ rot_coin [|1; 1; 1|] (3 - rtc !m);
			let p = sur !m in
				if sign angles p <> sign coins p then
				(* mouvement de transposition de deux angles ou de deux coins *)
					(let tr i j = sec (fun k -> if k = i then j else if k = j then i else k)
						in
							m := !m /*/ tr [|0; 1; 1|] [|1; 0; 1|];
					);
				(
					let sgnc = sign coins (sur !m)
					and n = (list_it (prefix +) (map (nqt !m) centres) 0) mod 2
					in
						if sgnc = 1 && n <> 0 || sgnc = - 1 && n = 0 then
							let f x = let ff = fun_of_mv1 !m in if x = [|1; 0; 0|] then ff x /./ rot x else ff x in
								m := mv1_of_fun f;
				);
				!m
;;

(* mélanger le cube par une suite aléatoire de mouvements de Rubik élémentaires *)
let melanger cube =
	let f = fun (OPS (a, b, c)) -> [a; b; c]
	and (ops1, ops2, ops3, ops4) = cube.rotations_faces1
	and t = make_vect 30 (fun () -> ())
	in
		(
			let v = vect_of_list (flat_map f [ops1; ops2; ops3; ops4])
			in
				for i = 0 to 29 do
					t.(i) <- v.(random__int 12);
				done;
				exe1 (list_of_vect t)
		);
;;


(*-------------------------------------------------------------------------------------------*)
(* 7 : COULEURS *)
(*-------------------------------------------------------------------------------------------*)

(* couleur rvb de la  couleur c *)
let couleur_rvb_de_couleur c =
	match c with
		| ROUGE -> graphics__red
		| ORANGE -> graphics__rgb 255 165 0
		| BLEU -> graphics__rgb 0 150 225
		| VERT -> graphics__green
		| JAUNE -> graphics__yellow
		| BLANC -> graphics__white
		| GRIS -> graphics__rgb 100 100 100
;;

(* association entre couleurs et vecteurs normaux aux faces du cube *)
let couleur_de_face v =
	match vect v with
		| 1, 0, 0 -> ORANGE
		| - 1, 0, 0 -> ROUGE
		| 0, 1, 0 -> VERT
		| 0, - 1, 0 -> BLEU
		| 0, 0, 1 -> BLANC
		| 0, 0, - 1 -> JAUNE
		| _ -> GRIS
;;

let couleur_rvb_de_face v =
	couleur_rvb_de_couleur (couleur_de_face v)
;;

let nom_couleur_de_face v =
	match vect v with
		| 1, 0, 0 -> "orange"
		| - 1, 0, 0 -> "rouge"
		| 0, 1, 0 -> "vert"
		| 0, - 1, 0 -> "bleu"
		| 0, 0, 1 -> "blanc"
		| 0, 0, - 1 -> "jaune"
		| _ -> "?"
;;

let nom_de_couleur couleur =
	match couleur with
		| ORANGE -> "ORANGE"
		| ROUGE -> "ROUGE"
		| VERT -> "VERT"
		| BLEU -> "BLEU"
		| BLANC -> "BLANC"
		| JAUNE -> "JAUNE"
		| _ -> "GRIS"
;;

(*-------------------------------------------------------------------------------------------*)
(* 8 : GRAPHISME *)
(*-------------------------------------------------------------------------------------------*)

let prj (ox, oy, ux, uy) v pt3 =
	let proj x y z =
		let c = sqrt 6. /. 2. in
			(c *. (y -. x) /. sqrt 2., c *. (-. (x +. y) +. 2. *. z) /. sqrt 6.)
	and (x, y, z) = vect (map_vect float_of_int pt3)
	in
		let (x1, y1, z1) =
			if v /|/ [|1; 1; 1|] = 1 then (x, y, z)
			else match vect v with
					| (_, 0, 0) -> (x -. 7., y, z)
					| (0, _, 0) -> (x, y -. 7., z)
					| _ -> (x, y, z -. 7.)
		in
			(
				int_of_float (float_of_int ox +. fst (proj x1 y1 z1) *. float_of_int ux),
				int_of_float (float_of_int oy +. snd (proj x1 y1 z1) *. float_of_int uy)
			)
;;

(* la fonction 'drawPoly' est utilisée pour tracer le pourtour des projections *)
(* des faces des minicubes *)
let drawPoly poly =
	let (x, y) = poly.(0) in graphics__moveto x y;
		for i = 1 to vect_length poly - 1 do
			let (x, y) = poly.(i) in graphics__lineto x y
		done;
		let (x, y) = poly.(0) in graphics__lineto x y;
;;

(* la fonction 'draw' est utilisée pour dessiner la projection 'x' d'une face *)
(* de minicube en superposant le tracé du pourtour à la couleur de remplissage *)
let draw x =
	let a, b = x in
		graphics__set_color b;
		graphics__fill_poly a;
		graphics__set_color graphics__black;
		drawPoly a
;;

(* 'face v c' renvoie, si le minicube à l'emplacement d'indice 'c' a une face F *)
(* dans la face du Rubik's cube normale au vecteur sortant 'v', sous forme de vecteur *)
(* une liste des 4 sommets de F correspondant à un parcours de son bord *)

let coeff = ref 2;;

let face v c =
	let e = v /|/ [|1; 1; 1|] in let w = [|e; e; e|] in
			let w1 = w /:/ rot v in
				let w2 = w1 /:/ rot v in
					let w3 = w2 /:/ rot v in
						let l = [w; w1; w2; w3] in
							let add m = for i = 0 to 2 do m.(i) <- m.(i) + !coeff * c.(i) done
							in
								do_list add l;
								vect_of_list l;
;;

(* 'faces' renvoie une liste de triplets : la première composante est un centre 'c', la deuxième composante *)
(* est un vecteur listant les 3 vecteurs unitaires sortants normaux aux faces visibles du minicube centré en 'c' *)
(* et la troisième est un vecteur dont la composante numéro i est un vecteur listant les 4 sommets de la face visible *)
(* normale au vecteur numéro i précédent : i = 0,1,2 pour un coin, i = 0,1 pour un angle, i = 0 pour un centre *)

let faces c =
	let d = vect_of_list (subtract (list_of_vect (diag c.(0) c.(1) c.(2))) [[|0; 0; 0|]]) in
		c, d, map_vect (fun v -> face v c) d
;;

let affiche1 plan mat context centre =
	let p = context.matrice in
		let _, d, f = faces centre in
			for i = 0 to vect_length d - 1 do
				let v = d.(i) /:/ mat in
					let g = map_vect (fun x -> x /:/ mat) f.(i) in
						draw ((map_vect (prj plan v)
								g),
							couleur_rvb_de_face (d.(i) /:/ transpose p));
			done
;;

(* affichage du cube 3x3, avec repères des centres, dans l'état mv : repère ADH *)
let tag plan mat context icentre =
	let f = face icentre icentre
	and v = icentre /:/ mat
	and milieu face =
		let aux v w = [|(v.(0) + w.(0)) / 2; (v.(1) + w.(1)) / 2; (v.(2) + w.(2)) / 2|] in
			aux face.(0) face.(2)
	in
		let g = map_vect (fun x -> x /:/ mat) f
		and m = milieu f
		in let p = context.matrice in
				let (c1, c2) = prj plan (icentre /:/ p) (m /:/ p)
				and (d1, d2) = prj plan (v /:/ p) (g.(0) /:/ p)
				in
					graphics__set_color graphics__black;
					graphics__moveto c1 c2;
					graphics__lineto d1 d2
;;

let affiche_mouvement plan context mv =
	let p = context.matrice in
		do_list (fun x -> affiche1 plan (transpose p /./ fun_of_mv1 mv x /./ p) context (x /:/ p)) indices;
		do_list (fun x -> tag plan (fun_of_mv1 mv x) context (x)) (centres)
;;

let dessine_cube cube = affiche_mouvement cube.repere1.plan cube.context1 cube.mouvement1.mv1;;

(*-------------------------------------------------------------------------------------------*)
(* 9 : RÉSOLUTION DU CUBE 3x3 par niveaux : "supérieur, médian, inférieur" *)
(*-------------------------------------------------------------------------------------------*)

exception Orienter_les_coins;;
exception Placer_angle_frontal_haut;;
exception Descendre_coin;;
exception Remonter_coin;;
exception Remonter_angle;;
exception Orienter_les_angles;;
exception Placer_les_angles;;
exception Placer_les_coins;;

let nom_position_de_face v =
	match vect v with
		| (1, 0, 0) -> "a"
		| (0, 1, 0) -> "d"
		| (0, 0, 1) -> "h"
		| (- 1, 0, 0) -> "p"
		| (0, - 1, 0) -> "g"
		| (0, 0, - 1) -> "b"
		| _ -> failwith "nom_position_de_face"
;;

let associe mat s =
	let adh = map_vect nom_position_de_face (mat)
	and pgb = map_vect nom_position_de_face ((- 1) /../ mat)
	in
		match s with
			| "a" -> adh.(0)
			| "d" -> adh.(1)
			| "h" -> adh.(2)
			| "p" -> pgb.(0)
			| "g" -> pgb.(1)
			| "b" -> pgb.(2)
			| "a'" -> adh.(0) ^ "'"
			| "d'" -> adh.(1) ^ "'"
			| "h'" -> adh.(2) ^ "'"
			| "p'" -> pgb.(0) ^ "'"
			| "g'" -> pgb.(1) ^ "'"
			| "b'" -> pgb.(2) ^ "'"
			| _ -> failwith "associe"
;;

(*-------------------------------------------------*)

(* en repère adh, la matrice de passage dans le groupe du cube telle que dans le repère adh associé *)
(* le coin centré en x dans l'état mv ait les couleurs adh coul1, coul2, coul3 *)
let context_adh_aux (coul1, coul2, coul3) mv xx =
	let couleurs_adh context mouvement x =
		let eclate x = [|[|x.(0); 0; 0|]; [|0; x.(1); 0|]; [|0; 0; x.(2)|]|] in
			let p = context.matrice in
				let m = eclate x /./ transpose p /./ (fun_of_mv1 (inverse mouvement)) (x /:/ transpose p) in
					map_vect couleur_de_face m
	in
		{matrice = hd (select (fun p -> couleurs_adh {matrice = p} mv xx = [|coul1; coul2; coul3|]) groupe_du_cube)}
;;

let context_adh (coul1, coul2, coul3) mv = context_adh_aux (coul1, coul2, coul3) mv [|1; 1; 1|];;

(*-------------------------------------------------*)

(* les couleurs en repère adh des faces antérieure, droite et gauche du cube non mélangé *)
let couleurs_faces_adh cube =
	let _ = cube.context1.matrice in
		let feminin adjectif = match adjectif with
				| "blanc" -> "blanche"
				| "vert" -> "verte"
				| "bleu" -> "bleue"
				| _ -> adjectif
		in
			let noms = map_vect (fun x -> feminin (nom_couleur_de_face x)) (transpose cube.context1.matrice) in
				printf__sprintf "Résultat avec :\nface antérieure %s\nface droite %s \nface haute %s" noms.(0) noms.(1) noms.(2)
;;

(* les couleurs en repère adh des faces visibles du minicube centré en x *)
let couleurs_adh cube x =
	let eclate x = [|[|x.(0); 0; 0|]; [|0; x.(1); 0|]; [|0; 0; x.(2)|]|] in
		let p = cube.context1.matrice and mv1 = cube.mouvement1.mv1 in
			let m = eclate x /./ transpose p /./ (fun_of_mv1 (inverse mv1)) (x /:/ transpose p) in
				map_vect couleur_de_face m
;;

(* les couleurs en repère adh des faces visibles du minicube centré en [|1; 1; 1|] *)
(* dans l'ordre antérieure, droite, haute *)
let couleurs_coin_adh cube = couleurs_adh cube [|1; 1; 1|];;

(* impression des noms des couleurs en repère adh des faces visibles du minicube centré en [|1; 1; 1|] *)
(* dans l'ordre antérieure, droite, haute *)
let noms_couleurs_coin_adh cube =
	let noms = map_vect nom_de_couleur (couleurs_adh cube [|1; 1; 1|])
	in
		printf__sprintf "(%s, %s, %s)" noms.(0) noms.(1) noms.(2)
;;

(*-------------------------------------------------*)

let affiche_mvt repere context mv1 = affiche_mouvement repere.plan context mv1;;

let nouveau_cube mouvement context repere anime =
	let dessine () = if !anime then affiche_mvt repere context mouvement.mv1
	in
		let rotations_faces () =
			let fct x () =
				let t = x /:/ transpose context.matrice in
					mouvement.mv1 <- mouvement.mv1 /*/ rub t;
					dessine ()
			and fct' x () =
				let t = x /:/ transpose context.matrice in
					mouvement.mv1 <- mouvement.mv1 /*/ rub' t;
					dessine ()
			in
				let (a, d, h) = vect (map_vect fct id)
				and (a', d', h') = vect (map_vect fct' id)
				and (p, g, b) = vect (map_vect fct idm)
				and (p', g', b') = vect (map_vect fct' idm)
				in (OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b'))
		and rotations_cube () =
			let rotate pp () = context.matrice <- context.matrice /./ pp;
				dessine ()
			in
				let (a, d, h) = vect (map_vect rotate (map_vect rot id))
				and (a', d', h') = vect (map_vect rotate (map_vect rot' id))
				in
					(OPS (a, d, h), OPS (a', d', h'))
		
		in
			{
				anime1 = anime;
				mouvement1 = mouvement;
				context1 = context;
				repere1 = repere;
				mvi = {mv1 = mv1_of_fun (fun x -> if est_centre x then id else matrice_nulle)};
				rotations_cube1 = rotations_cube ();
				rotations_faces1 = rotations_faces ();
				boutons1 = make_vect 1 {titre = ""; orx = 0; ory = 0; largeur = 0;
					hauteur = 0; couleur = BLANC; action = (fun () -> ()); bas = false};
			}
;;

let resoudre_le_cube cube completement =
	(* (y, m) = pos0 x : le minicube d'indice x est à l'emplacement d'indice y et m est sa matrice *)
	(* de déplacement (telle que y=xm) (repère ADH) *)
	let context = cube.context1
	and mouvement = cube.mouvement1
	in
		let nbqt = ref 0
		and lo = ref ""
		and matr = cube.context1.matrice
		and mctx = ref id
		in
			let (pos0, pos1) =
				let pos mv1 x =
					let m = let p = cube.context1.matrice in fun_of_mv1 (inverse (cst p) /*/ mv1 /*/ cst p) (x)
					in
						x /:/ m, m
				in
					(fun cube x -> pos cube.mouvement1.mv1 x),
					(fun cube x -> pos (inverse mouvement.mv1) x)
			and
			(OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b')) = cube.rotations_faces1
			and
			(OPS (a0, d0, h0), OPS (a0', d0', h0')) = cube.rotations_cube1
			in
				let op_with_name s =
					let la = [("a", a); ("p", p); ("h", h); ("b", b); ("d", d); ("g", g);
							("a'", a'); ("p'", p'); ("h'", h'); ("b'", b'); ("d'", d'); ("g'", g');
							("a0", a0); ("d0", d0); ("h0", h0); ("a0'", a0'); ("d0'", d0'); ("h0'", h0');]
					in
						assoc s la
				in
					let exec1 str =
						let temp = cube.context1.matrice in
							let listop = op_names_from_string str in
								exe1 (map op_with_name listop);
								let temp1 = cube.context1.matrice in
									mctx := !mctx /./ transpose temp /./ temp1;
					and exec str =
						let listop = op_names_from_string str in
							nbqt := !nbqt + list_length listop;
							if !mctx <> id then
								(
									lo := !lo ^ format_string (decomposition (transpose !mctx));
									mctx := id
								);
							exe1 (map op_with_name listop);
							lo := !lo ^ (format_string str);
							let m = (transpose cube.context1.matrice) /./ matr in
								print_string (format_string (concat (map (associe m) listop)))
					in
						
						let niveau_superieur () =
							(* niveau supérieur *)
							let orienter_le_centre () =
								let n = nqt mouvement.mv1 ([|0; 0; 1|] /:/ transpose context.matrice) in
									if n > 0 then (
											print_string "(* ORIENTATION DU CENTRE *)\n";
											lo := !lo ^ "(* ORIENTATION DU CENTRE *)\n";
										);
									if n = 1 then exec "h'"
									else if n = 2 then exec "hh"
									else if n = 3 then exec "h"
							and placer_et_orienter_les_angles () =
								let placer_angle_frontal_haut () =
									let v, _ = pos0 cube [|1; 0; 1|] in
										match vect v with
											| (1, 0, 1) -> ()
											| (1, 1, 0) -> exec "a'"
											| (1, 0, - 1) -> exec "aa"
											| (1, - 1, 0) -> exec "a"
											| (0, 1, 1) -> exec "d'a'"
											| (- 1, 1, 0) -> exec "h'd'h"
											| (0, 1, - 1) -> exec "da'd'"
											| (0, - 1, 1) -> exec "ga"
											| (- 1, - 1, 0) -> exec "hgh'"
											| (0, - 1, - 1) -> exec "g'ag"
											| (- 1, 0, 1) -> exec "ppbbaa"
											| (- 1, 0, - 1) -> exec "bbaa"
											| _ -> raise Placer_angle_frontal_haut
								and mal_oriente () =
									(snd (pos0 cube [|1; 0; 1|])).(2) <> [|0; 0; 1|]
								in
									for i = 0 to 3 do
										placer_angle_frontal_haut ();
										if mal_oriente () then exec "h'd'ha'";
										exec1 "h0"
									done
							and placer_et_orienter_les_coins () =
								let descendre_coin () =
									let w, m = pos0 cube [|1; 1; 1|] in
										if (w = [|1; 1; 1|]) && (m = id) then ()
										else
											match vect w with
												| (- 1, 1, 1) -> exec "p'b'p"
												| (- 1, - 1, 1) -> exec "pbbp'"
												| (1, - 1, 1) -> exec "gbg'"
												| (1, 1, 1) -> exec "aba'b'"
												| (- 1, 1, - 1) -> exec "b'"
												| (- 1, - 1, - 1) -> exec "bb"
												| (1, - 1, - 1) -> exec "b"
												| (1, 1, - 1) -> ()
												| _ -> raise Descendre_coin
								and remonter_coin () =
									let (v, m) = pos0 cube [|1; 1; 1|] in
										if (v = [|1; 1; 1|]) && (m = id) then ()
										else
											let w = m.(2) in match vect w with
													| (1, 0, 0) -> exec "da'd'a"
													| (0, 1, 0) -> exec "a'dad'"
													| (0, 0, - 1) -> exec "ab'a'bbda'd'a"
													| _ -> raise Remonter_coin
								in
									for i = 0 to 3 do
										descendre_coin ();
										remonter_coin ();
										exec1 "h0"
									done;
							in
								if completement then orienter_le_centre ();
								print_string "(* LA CROIX *)\n";
								lo := !lo ^ "(* LA CROIX *)\n";
								placer_et_orienter_les_angles ();
								print_string "(* LES COINS *)\n";
								lo := !lo ^ "(* LES COINS *)\n";
								placer_et_orienter_les_coins ();
						
						and niveau_median () =
							(* niveau médian *)
							let orienter_les_centres_lateraux () =
								print_string "(* ORIENTATION DES CENTRES *)\n";
								lo := !lo ^ "(* ORIENTATION DES CENTRES *)\n";
								let aux () =
									let n = nqt mouvement.mv1 ([|1; 0; 0|] /:/ transpose context.matrice)
									in
										if n = 1 then
											exec "aabba'bbaa"
										else if n = 2 then
											exec "aabbaabbaa"
										else if n = 3 then
											exec "aabbabbaa"
								in
									(aux (); exec1 "h0"; aux (); exec1 "h0"; aux (); exec1 "h0"; aux (); exec1 "h0");
									print_string "(* ORIENTATION DES CENTRES TERMINÉE *)\n";
									lo := !lo ^ "(* ORIENTATION DES CENTRES TERMINÉE *)\n";
							and placer_angle_frontal_droit () =
								let descendre_angle () =
									let aux () = exec "bab'a'b'd'bd" in
										let x, _ = pos0 cube [|1; 1; 0|] in
											match vect x with
												| (1, 1, 0) -> aux ()
												| (- 1, 1, 0) -> exec1 "h0"; aux (); exec1 "h0'"
												| (- 1, - 1, 0) -> exec1 "h0"; exec1 "h0"; aux (); exec1 "h0"; exec1 "h0";
												| (1, - 1, 0) -> exec1 "h0'"; aux (); exec1 "h0"
												| _ -> ()
								and remonter_angle () =
									let aux_r () = exec "b'd'bdbab'a'"
									and aux_l () = exec "bab'a'b'd'bd"
									in
										let x, m = pos0 cube [|1; 1; 0|] in
											if m.(0) <> [|0; 0; - 1|] then
												match vect x with
													| (1, 0, - 1) -> aux_r ()
													| (0, - 1, - 1) -> exec "b"; aux_r ()
													| (- 1, 0, - 1) -> exec "bb"; aux_r ()
													| (0, 1, - 1) -> exec "b'"; aux_r ()
													| _ -> raise Remonter_angle
											else
												match vect x with
													| (1, 0, - 1) -> exec "b"; aux_l ()
													| (0, - 1, - 1) -> exec "bb"; aux_l ()
													| (- 1, 0, - 1) -> exec "b'"; aux_l ()
													| (0, 1, - 1) -> aux_l ()
													| _ -> raise Remonter_angle
								in
									let x, m = pos0 cube [|1; 1; 0|] in
										if (x = [|1; 1; 0|]) && (m = id) then ()
										else (
												descendre_angle ();
												remonter_angle ()
											)
							in
								if completement then orienter_les_centres_lateraux ();
								for i = 0 to 3 do
									placer_angle_frontal_droit ();
									exec1 "h0"
								done
						
						and niveau_inferieur () =
							(* niveau inférieur *)
							let orienter_les_angles () =
								let est_mal_oriente angle =
									let (_, m) = pos1 cube angle in
										m.(2) <> [|0; 0; 1|]
								in
									let v = map_vect est_mal_oriente
										[|[|1; 0; - 1|]; [|0; - 1; - 1|]; [|- 1; 0; - 1|]; [|0; 1; - 1|]|]
									in match (v.(0), v.(1), v.(2), v.(3)) with
											| (false, false, false, false) -> ()
											| (true, true, true, true) -> exec "dbab'a'd'bdaba'b'd'"
											
											| (false, false, true, true) -> exec1 "h0"; exec "dbab'a'd'"
											| (true, false, false, true) -> exec "dbab'a'd'"
											| (true, true, false, false) -> exec1 "h0'"; exec "dbab'a'd'"
											| (false, true, true, false) -> exec1 "h0"; exec1 "h0"; exec "dbab'a'd'"
											
											| (false, _, false, _) -> exec "daba'b'd'"
											| (_, false, _, false) -> exec1 "h0"; exec "daba'b'd'"
											| _ -> raise Orienter_les_angles
							
							and placer_les_angles () =
								if completement then
								(* on fait en sorte que le nombre de quarts de tours soit nul modulo 4 *)
								(* la permutation des coins devrait alors être paire *)
								(* de même que la permutation des angles *)
									(
										let n = nqt mouvement.mv1 ([|0; 0; - 1|] /:/ transpose context.matrice) in
											if n = 1 then exec "b'" else if n = 3 then exec "b" else if n = 2 then exec "bb"
									)
								else
								(* on fait en sorte que la permutation des angles soit paire *)
									(
										if sign angles (sur mouvement.mv1) = - 1 then
											(
												print_string "(* CONTRÔLE DE PARITÉ *)\n";
												lo := !lo ^ "(* CONTRÔLE DE PARITÉ *)\n";
												exec "b";
												print_string "(* FIN DE CONTRÔLE DE PARITÉ *)\n";
												lo := !lo ^ "(* FIN DE CONTRÔLE DE PARITÉ *)\n";
											)
									);
								let permuter () =
									(* laisse fixe l'angle arrière et permute circulairement les autres *)
									(* dans le sens direct vu d'en bas *)
									exec "dbbd'b'db'd'"
								and permuter' () =
									(* laisse fixe l'angle arrière et permute circulairement les autres *)
									(* dans le sens indirect vu d'en bas *)
									exec "dbd'bdbbd'"
								in
									let chercher_un_angle_bien_place () =
										let i = ref 0 in
											while !i < 4 && fst (pos0 cube [|- 1; 0; - 1|]) <> [|- 1; 0; - 1|] do
												exec1 "h0";
												incr i
											done;
											!i
									in
										let j = chercher_un_angle_bien_place () in
											if j = 4 (* aucun angle bien placé *) then
												(
													permuter ();
													let _ = chercher_un_angle_bien_place () in ()
												)
											else ();
											let v, _ = pos0 cube [|1; 0; - 1|] in match vect v with
													| (0, - 1, - 1) -> permuter ()
													| (0, 1, - 1) -> permuter' ()
													| (1, 0, - 1) -> ()
													| _ -> raise Placer_les_angles
							
							and placer_les_coins () =
								(* à ce stade la permutation des coins devrait être paire *)
								let permuter () =
									(* laisse fixe le coin frontal droit et permute circulairement *)
									(* les autres dans le sens direct vu d'en bas *)
									exec "bab'p'ba'b'p"
								and permuter' () =
									(* laisse fixe le coin frontal droit et permute circulairement *)
									(* les autres dans le sens indirect vu d'en bas *)
									exec "p'bab'pba'b'"
								in
									let chercher_un_coin_bien_place () =
										let i = ref 0 in
											while !i < 4 && fst (pos0 cube [|1; 1; - 1|]) <> [|1; 1; - 1|] do
												exec1 "h0";
												incr i
											done;
											!i
									in
										let j = chercher_un_coin_bien_place () in
											
											if j = 4 (* aucun coin bien placé *) then (
													permuter ();
													let _ = chercher_un_coin_bien_place () in ()
												)
											else ();
											let v, _ = pos0 cube [|- 1; - 1; - 1|] in match vect v with
													| (1, - 1, - 1) -> permuter ()
													| (- 1, 1, - 1) -> permuter' ()
													| (- 1, - 1, - 1) -> ()
													| _ -> raise Placer_les_coins
							
							and orienter_les_coins () =
								let faire_tourner () =
									(* fait tourner les coins frontaux inférieurs sur eux-mêmes: *)
									(* le coin gauche dans le sens direct, le coin droit en sens inverse *)
									exec "p'b'pb'p'bbp";
									exec "aba'babba'"
								and
								faire_tourner' () =
									(* fait tourner les coins frontaux inférieurs sur eux-mêmes: *)
									(* le coin droit dans le sens direct, le coin gauche en sens inverse *)
									exec "abba'b'ab'a'";
									exec "p'bbpbp'bp"
								in
									let orienter_frontal_inferieur_droit () =
										let _, m = pos0 cube [|1; 1; - 1|] in
											let v = m.(2) in
												match vect v with
													| (0, 0, 1) -> ()
													| (- 1, 0, 0) -> faire_tourner' ()
													| (0, - 1, 0) -> faire_tourner ()
													| _ -> raise Orienter_les_coins
									in
										for i = 0 to 2 do
											orienter_frontal_inferieur_droit ();
											exec1 "h0'"
										done
							in
								print_string "(* LA CROIX *)\n";
								lo := !lo ^ "(* LA CROIX *)\n";
								orienter_les_angles ();
								placer_les_angles ();
								print_string "(* LES COINS *)\n";
								lo := !lo ^ "(* LES COINS *)\n";
								placer_les_coins ();
								orienter_les_coins ();
								print_newline ();
								(* pour ramener le cube à sa position initiale *)
								let ss = decomposition (transpose (!mctx /./ transpose cube.context1.matrice /./ matr)) in
									if ss <> "" then
										(
											lo := !lo ^ "(* - Ramener le cube à sa position initiale *)\n";
											lo := !lo ^ format_string ss;
										);
						in
							try
								let mat = cube.context1.matrice
								and noms_couleurs = noms_couleurs_coin_adh cube
								in
									print_string "include \"exemples/Caml Light/Rubik/interfaces/interface3.ml\";;\n\n";
									printf__printf "let mv = lire_mouv \"mouv3333\";;\n";
									printf__printf "let ctx = context_adh %s mv;;\n" noms_couleurs;
									printf__printf "graphics__open_graph \" 612x612\";;\n\n";
									
									printf__printf "(* coin Antérieur Droit Haut : %s *)\n\n" noms_couleurs;
									print_string "(* RÉSOLUTION SANS ROTATIONS GLOBALES *)\n\n";
									printf__printf "(* %s *)\n\n" (couleurs_faces_adh cube);
									
									printf__printf "cube.context1.matrice <- ctx.matrice;;\n";
									printf__printf "cube.mouvement1.mv1 <- mv;;\n";
									printf__printf "dessine_cube ctx mv;;\n\n";
									
									cube.context1.matrice <- mat;
									dessine_cube cube;
									printf__printf "\n(* NIVEAU SUPÉRIEUR *)\n";
									lo := !lo ^ "\n(* NIVEAU SUPÉRIEUR *)\n";
									niveau_superieur ();
									printf__printf "\n(* NIVEAU MÉDIAN *)\n";
									lo := !lo ^ "\n(* NIVEAU MÉDIAN *)\n";
									niveau_median ();
									printf__printf "\n(* NIVEAU INFÉRIEUR *)\n";
									lo := !lo ^ "\n(* NIVEAU INFÉRIEUR *)\n";
									niveau_inferieur ();
									
									cube.context1.matrice <- mat;
									dessine_cube cube;
									print_newline ();
									
									print_string "(* RÉSOLUTION AVEC ROTATIONS GLOBALES *)\n\n";
									printf__printf "cube.context1.matrice <- ctx.matrice;;\n";
									printf__printf "cube.mouvement1.mv1 <- mv;;\n";
									printf__printf "dessine_cube ctx mv;;\n\n";
									
									print_string !lo;
									print_newline ();
									!nbqt
							with
								| Orienter_les_coins ->
											print_string "erreur dans orienter_les_coins\n"; !nbqt
								| Placer_les_coins ->
											print_string "erreur dans placer_les_coins\n"; !nbqt
								| Placer_les_angles ->
											print_string "erreur dans placer_les_angles\n"; !nbqt
								| Orienter_les_angles ->
											print_string "erreur dans orienter_les_angles\n"; !nbqt
;;

(*-------------------------------------------------------------------------------------------*)
(* 10 : BOUTONS *)
(*-------------------------------------------------------------------------------------------*)
(* Gestion par boutons des mouvements globaux et des mouvements de Rubik *)

let bouton titre orx ory largeur hauteur couleur action =
	{titre = titre; orx = orx; ory = ory; hauteur = hauteur; largeur = largeur;
		couleur = couleur; action = action; bas = false}
;;

let inverse_bouton b =
	b.bas <- true;
	graphics__set_color graphics__black;
	graphics__fill_rect b.orx b.ory b.largeur b.hauteur;
	let (x, y) = graphics__text_size b.titre in
		graphics__moveto (b.orx + (b.largeur - x) / 2) (b.ory + (b.hauteur - y) / 2);
		graphics__set_color graphics__white;
		graphics__draw_string b.titre;
		let p = make_vect 4 (0, 0) in
			p.(0) <- (b.orx, b.ory);
			p.(1) <- (b.orx + b.largeur, b.ory);
			p.(2) <- (b.orx + b.largeur, b.ory + b.hauteur);
			p.(3) <- (b.orx, b.ory + b.hauteur);
			graphics__set_color graphics__black;
			drawPoly p
;;

let dessine_bouton b =
	b.bas <- false;
	graphics__set_color (couleur_rvb_de_couleur (b.couleur));
	graphics__fill_rect b.orx b.ory b.largeur b.hauteur;
	let (x, y) = graphics__text_size b.titre in
		graphics__moveto (b.orx + (b.largeur - x) / 2) (b.ory + (b.hauteur - y) / 2);
		graphics__set_color graphics__black;
		graphics__draw_string b.titre;
		let p = make_vect 4 (0, 0) in
			p.(0) <- (b.orx, b.ory);
			p.(1) <- (b.orx + b.largeur, b.ory);
			p.(2) <- (b.orx + b.largeur, b.ory + b.hauteur);
			p.(3) <- (b.orx, b.ory + b.hauteur);
			graphics__set_color graphics__black;
			drawPoly p
;;

let set_action bouton action =
	bouton.action <- action
;;

let gestion_bouton bouton mouse_down mousex mousey =
	if bouton.orx < mousex && mousex < bouton.orx + bouton.largeur
		&& bouton.ory < mousey && mousey < bouton.ory + bouton.hauteur then (
			if mouse_down then (
					if not bouton.bas then (
							inverse_bouton bouton;
						)
				)
			else (
					if bouton.bas then (
							dessine_bouton bouton;
							bouton.action ()
						)
				)
		)
	else (
			if bouton.bas then (
					dessine_bouton bouton
				)
		)
;;

(*-------------------------------------------------------------------------------------------*)
(* 11 : RANGÉE DE BOUTONS POUR LA MANIPULATION DU CUBE 3x3 *)
(*-------------------------------------------------------------------------------------------*)

(* Gestion par boutons des mouvements globaux et des mouvements de Rubik *)
(* Pour fenêtre de largeur 612 et hauteur 612 *)
(* Cube avec en bas de fenêtre une rangée de 18 boutons de largeur 34 : 18 x 34 = 612 *)

(* Fenêtre de largeur 612 et hauteur 612 : origine  au centre (306,306), unités : 20,20 *)

(* Rangée de 18 boutons en bas de fenêtre pour les mouvements globaux et les rotations des faces externes *)
let dessine_boutons1 cube =
	let couleur_titre titre =
		let face titre = match titre with
				| "A" | "A'" | "a" | "a'" -> [|1; 0; 0|]
				| "D" | "D'" | "d" | "d'" -> [|0; 1; 0|]
				| "H" | "H'" | "h" | "h'" -> [|0; 0; 1|]
				| "p" | "p'" -> [|- 1; 0; 0|]
				| "g" | "g'" -> [|0; - 1; 0|]
				| "b" | "b'" -> [|0; 0; - 1|]
				| _ -> failwith "face"
		in
			couleur_de_face ((face titre) /:/ transpose cube.context1.matrice)
	in
		for i = 0 to vect_length cube.boutons1 - 1 do
			cube.boutons1.(i).couleur <- couleur_titre cube.boutons1.(i).titre;
			dessine_bouton cube.boutons1.(i)
		done
;;

let cree_boutons1 cube =
	let titres =
		[|"A"; "A'"; "H"; "H'"; "D"; "D'";
			"a"; "a'"; "h"; "h'"; "d"; "d'";
			"p"; "p'"; "b"; "b'"; "g"; "g'"
		|]
	in
		let set_actions boutons =
			let (OPS (a0, d0, h0), OPS (a0', d0', h0')) = cube.rotations_cube1
			and (OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b')) =
				cube.rotations_faces1
			in
				let v = [|a0; a0'; h0; h0'; d0; d0'; a; a'; h; h'; d; d'; p; p'; b; b'; g; g'|]
				and op_names = [|"a0"; "a0'"; "h0"; "h0'"; "d0"; "d0'"; "a"; "a'"; "h"; "h'"; "d"; "d'"; "p"; "p'"; "b"; "b'"; "g"; "g'"|]
				in
					for i = 0 to 5 do
						set_action boutons.(i) (fun () -> v.(i) (); print_string (op_names.(i));
											dessine_cube cube;
											dessine_boutons1 cube;
											flush std_out)
					done;
					for i = 6 to vect_length v - 1 do
						set_action boutons.(i) (fun () -> v.(i) (); print_string (op_names.(i));
											dessine_cube cube; flush std_out)
					done
		in
			let n = vect_length titres in
				let boutons = make_vect n (bouton "" 0 0 0 0 BLANC (fun () -> ())) in
					for i = 0 to n - 1 do
						boutons.(i) <- {titre = titres.(i); orx = i * 34; ory = 0; hauteur = 30;
							largeur = 34; couleur = BLANC; action = (fun () -> ()); bas = false}
					done;
					set_actions boutons;
					cube.boutons1 <- boutons;
;;

let gestion_boutons1 cube is_down mousex mousey =
	for i = 0 to vect_length cube.boutons1 - 1 do
		gestion_bouton cube.boutons1.(i) is_down mousex mousey
	done;
;;

exception Quitter;;

let cree_boutons cube actions =
	let largeur1, hauteur1 = graphics__text_size "Quitter"
	and largeur, hauteur = graphics__text_size "Résoudre"
	and largeur3, hauteur3 = graphics__text_size "Résoudre complètement"
	and largeur4, hauteur4 = graphics__text_size "Mélanger"
	and largeur2, hauteur2 = graphics__text_size "Restituer"
	and largeur5, hauteur5 = graphics__text_size "Composer"
	and _ = cree_boutons1 cube;
	in
		let
		bouton_quitter = bouton "Quitter" (graphics__size_x () - largeur1 - 20) (graphics__size_y () - hauteur1 - 20)
			(largeur1 + 10) (hauteur1 + 10) JAUNE (fun () -> raise Quitter)
		and bouton_resoudre = bouton "Résoudre" 20 (graphics__size_y () - hauteur - 20)
			(largeur + 10) (hauteur + 10) JAUNE actions.(0)
		and bouton_resoudre_completement = bouton "Résoudre complètement" 20 (graphics__size_y () - 2 * (hauteur + 20))
			(largeur3 + 10) (hauteur3 + 10) JAUNE actions.(1)
		and bouton_melanger = bouton "Mélanger" ((graphics__size_x () - largeur4) / 2 - 10) (graphics__size_y () - hauteur4 - 20)
			(largeur4 + 10) (hauteur4 + 10) JAUNE actions.(2)
		and bouton_restaurer = bouton "Restituer" ((graphics__size_x () - largeur2) / 2 - 10) (graphics__size_y () - 2 * (hauteur2 + 20))
			(largeur2 + 10) (hauteur2 + 10) JAUNE actions.(3)
		and bouton_composer = bouton "Composer" ((graphics__size_x () - largeur5) / 2 - 10) (graphics__size_y () - 3 * (hauteur5 + 20))
			(largeur5 + 10) (hauteur5 + 10) JAUNE actions.(4)
		in
			[bouton_quitter; bouton_resoudre; bouton_resoudre_completement; bouton_melanger; bouton_restaurer; bouton_composer]
;;

let dessine_boutons liste_boutons =
	do_list dessine_bouton liste_boutons
;;

let gestion_boutons liste_boutons is_down mousex mousey =
	do_list (fun b -> gestion_bouton b is_down mousex mousey) liste_boutons
;;

(*-------------------------------------------------------------------------------------------*)
(* 12 : SAISIE D'UN MOUVEMENT DU CUBE *)
(*-------------------------------------------------------------------------------------------*)

exception Fin_de_recherche;;(* on a cliqué sur une face de minicube dans le panneau gauche *)

(* affichage en gris d'un minicube *)
let griser plan centre =
	let c, d, f = faces centre in
		for i = 0 to vect_length d - 1 do
			let v = d.(i) in
				draw ((map_vect (prj plan v)
						f.(i)),
					couleur_rvb_de_couleur GRIS)
		done
;;

let est_vide_a_gauche cube v = not (est_centre v) && fun_of_mv1 cube.mvi.mv1 v <> matrice_nulle;;

let translate (ox, oy, ux, uy) h = (ox + h, oy, ux, uy);;

let affiche_mvi cube plan largeur =
	let plan1 = translate plan (largeur / 2) in
		do_list (griser plan1) (select (fun i -> not est_centre i) indices);
		let p = cube.context1.matrice in
			let f i =
				if est_vide_a_gauche cube i then griser plan (i /:/ p)
				else (affiche1 plan id cube.context1 (i /:/ p));
				if fun_of_mv1 cube.mvi.mv1 i <> matrice_nulle || est_centre i then
					affiche1 plan1 (transpose p /./ fun_of_mv1 cube.mvi.mv1 i /./ p) cube.context1 (i /:/ p)
			in
				do_list f indices;
				do_list (fun x -> tag plan1 (fun_of_mv1 cube.mvi.mv1 x) cube.context1 (x)) (centres)
;;

(* pour les clics dans les faces des minicubes *)
(* est_dans_poly p x renvoie 'true' ssi le point 'x' est intérieur au quadrilatère convexe 'p'*)
let est_dans_poly p x =
	let prefix /-/ (a1, b1) (a2, b2) = (a1 - a2, b1 - b2)
	and det (a1, b1) (a2, b2) = a1 * b2 - a2 * b1
	in
		let p0 = p.(0) /-/ x
		and p1 = p.(1) /-/ x
		and p2 = p.(2) /-/ x
		and p3 = p.(3) /-/ x
		in
			det p0 p1 * det p1 p2 > 0
			&& det p1 p2 * det p2 p3 > 0
			&& det p2 p3 * det p3 p0 > 0
;;

let gestion_gauche_droite cube largeur =
	let choix_a_gauche = ref NIL and choix_a_droite = ref NIL
	and action plan mousex mousey =
		let x = ref NIL in
			try
				do_list (
					fun i ->
									let (c, d, f) = faces i in
										for j = 0 to vect_length d - 1 do
											let face = map_vect (prj plan d.(j)) f.(j) in
												if est_dans_poly face (mousex, mousey) then (x := COUPLE (c, d.(j)); raise Fin_de_recherche)
										done
				)
				(select (fun i -> not (est_centre i)) indices);
				NIL
			with Fin_de_recherche -> !x
	in
		let gestion_a_gauche plan mousex mousey = (
				if mousex < largeur / 2 then (
						choix_a_gauche := action plan mousex mousey;
					)
			)
		and gestion_a_droite plan mousex mousey = (
				if mousex >= largeur / 2 then
					(
						choix_a_droite := action (translate plan (largeur / 2)) mousex mousey;
						let p = cube.context1.matrice in
							match !choix_a_droite with
								| COUPLE (w1, w2) ->
											(
												let est_vide_a_droite v = for_all (fun i -> i /:/ fun_of_mv1 cube.mvi.mv1 i <> v) indices
												in
													if est_vide_a_droite (w1 /:/ transpose p) then
														(
															if not est_centre w1 then
																(match !choix_a_gauche with
																		| COUPLE (v1, v2) ->
																					(try
																							let mat = hd (select (fun m -> v1 /:/ m = w1 && v2 /:/ m = w2) groupe_du_cube)
																							in
																								(let f i = (*if est_centre i then id else *)if est_centre i then fun_of_mv1 cube.mvi.mv1 i else if i /:/ p <> v1 then fun_of_mv1 cube.mvi.mv1 i else p /./ mat /./ transpose p in
																										cube.mvi.mv1 <- map (fun i -> (i, f i)) indices;
																								);
																						with Failure "hd" -> ());
																					choix_a_gauche := NIL
																		| NIL -> ()
																);
														)
													else
														(
															cube.mvi.mv1 <- map (fun i -> (i, let m = fun_of_mv1 cube.mvi.mv1 i in if i /:/ m /:/ p = w1 then matrice_nulle else m)) indices;
															choix_a_gauche := NIL;
														)
											)
								| NIL -> ()
					);
			)
		in
			(gestion_a_gauche, gestion_a_droite)
;;

let gestion_centres cube plan largeur mousex mousey =
	do_list (fun i ->
						let (c, d, f) = faces i in
							for j = 0 to vect_length d - 1 do
								let face = map_vect (prj plan d.(j)) f.(j) and face1 = map_vect (prj (translate plan (largeur / 2)) d.(j)) f.(j) in
									if est_dans_poly face (mousex, mousey) then (cube.mvi.mv1 <- ajuste_centre cube.mvi.mv1 rot' (c /:/ transpose cube.context1.matrice))
									else if est_dans_poly face1 (mousex, mousey) then (cube.mvi.mv1 <- ajuste_centre cube.mvi.mv1 rot (c /:/ transpose cube.context1.matrice));
							done
	)
	(select est_centre indices);
;;



(*-------------------------------------------------------------------------------------------*)
(* 13 : BOUCLE DE SAISIE DU MOUVEMENT *)
(*-------------------------------------------------------------------------------------------*)

let tout_vide_a_gauche cube = for_all (est_vide_a_gauche cube) (select (fun i -> not est_centre i) indices);;

let boucle_saisie cube s =
	graphics__open_graph s;
	let (largeur, hauteur) = (graphics__size_x ()), (graphics__size_y ())
	in
		let plan = (largeur / 4, hauteur / 2, largeur / 60, largeur / 60)
		in let (gestion_a_gauche, gestion_a_droite) = gestion_gauche_droite cube largeur
			in
				graphics__set_window_title "Composer un cube";
				affiche_mvi cube plan largeur;
				graphics__set_color graphics__black;
				graphics__moveto (largeur / 2) 0;
				graphics__lineto (largeur / 2) hauteur; (* cloison entre les deux panneaux *)
				printf__printf "\n-----------------------------------------------------------\n";
				printf__printf "COMPOSITION D'UN MOUVEMENT:\n\n";
				printf__printf "- Pour transférer un minicube de gauche à droite, cliquer sur une de ses faces puis cliquer à droite sur la destination de cette face (une face grise).\n";
				printf__printf "- Pour ramener un minicube de droite à sa place à gauche, cliquer sur une de ses faces.\n\n";
				printf__printf "- La rotation totale des minicubes est vérifiée quand tous les minicubes sont à droite.\n";
				printf__printf "- En cas de copie d'un cube correct, une rotation totale non nulle implique une erreur dans cette copie.\n\n";
				print_newline ();
				try
					let largeur1, hauteur1 = graphics__text_size "Quitter"
					in
						let bouton_quitter = bouton "Quitter" (graphics__size_x () - largeur1 - 20) (graphics__size_y () - hauteur1 - 20)
							(largeur1 + 10) (hauteur1 + 10) JAUNE (fun () -> raise Quitter)
						in
							dessine_bouton bouton_quitter;
							while true do
								let status = graphics__wait_next_event [graphics__Button_down] in
									let mousex = status.graphics__mouse_x and mousey = status.graphics__mouse_y
									in
										gestion_bouton bouton_quitter true mousex mousey;
										let status = graphics__wait_next_event [graphics__Button_up] in
											gestion_bouton bouton_quitter false mousex mousey;
											gestion_a_gauche plan mousex mousey;
											gestion_a_droite plan mousex mousey;
											gestion_centres cube plan largeur mousex mousey;
											affiche_mvi cube plan largeur;
											if tout_vide_a_gauche cube then
												(
													printf__printf "TOUS PLACÉS !\n";
													print_newline ();
													let rc = rtc cube.mvi.mv1 and ra = rta cube.mvi.mv1 in
														printf__printf "signature de la permutation des coins : %d\n" (sign (select est_coin indices) (sur cube.mvi.mv1));
														printf__printf "signature de la permutation des angles : %d\n" (sign (select est_angle indices) (sur cube.mvi.mv1));
														printf__printf "  ces signatures devraient être égales\n";
														printf__printf "nombre de quarts de tour des centres modulo 4 : %d\n" (nqt_total cube.mvi.mv1);
														printf__printf "  ce nombre devrait être 0 ou 2 si les signatures précédentes valent 1)\n";
														printf__printf "  il devrait être 1 ou 3 si ces signatures valent -1\n";
														printf__printf "rotation totale des coins : %d\n" rc;
														printf__printf "rotation totale des angles : %d\n" ra;
														printf__printf "  ces rotations totales devraient être nulles";
														print_newline ();
												);
							done;
				with Quitter ->
								graphics__close_graph ();
								if tout_vide_a_gauche cube then
									(
										cube.mouvement1.mv1 <- cube.mvi.mv1;
									)
								else (
										printf__printf "composition de mouvement inachevée";
										print_newline ();
									)
;;

(*-------------------------------------------------------------------------------------------*)
(* 14 : BOUCLE DE MANIPULATION DU CUBE 3x3 *)
(*-------------------------------------------------------------------------------------------*)

let boucle1 cube actions =
	try
		dessine_cube cube;
		let liste_boutons = cree_boutons cube actions
		in
			dessine_boutons liste_boutons;
			dessine_boutons1 cube;
			while true do
				let status = graphics__wait_next_event [graphics__Button_down; graphics__Button_up; graphics__Mouse_motion] in
					let mousex = status.graphics__mouse_x
					and mousey = status.graphics__mouse_y
					and is_down = status.graphics__button
					in
						gestion_boutons liste_boutons is_down mousex mousey;
						gestion_boutons1 cube is_down mousex mousey;
			done;
	with Quitter -> graphics__close_graph ()
;;

(*-------------------------------------------------------------------------------------------*)
(* 15 : BOUCLE PRINCIPALE *)
(*-------------------------------------------------------------------------------------------*)

let rec boucle cube =
	(let (ox, oy, _, _) = cube.repere1.plan in
			let (sx, sy) = (string_of_int (2 * ox), string_of_int (2 * oy)) in
				graphics__open_graph (" " ^ sx ^ "x" ^ sy);
				graphics__set_window_title "Résoudre le cube"
	);
	try
		boucle1 cube
		[|
			((* résolution simple *)
				fun () ->
								print_string "\n(*-------------------------------------------------------*)\n";
								print_string "(* RÉSOLUTION SIMPLE *)\n";
								print_newline ();
								enregistrer_mouv cube.mouvement1.mv1 "mouv3333";
								let l = resoudre_le_cube cube false in
									dessine_cube cube;
									printf__printf "\n(* nombre de quarts de tours : %d *)\n" l;
									print_newline ()
			);
			((* résolution complète *)
				fun () ->
								print_string "\n(*-------------------------------------------------------*)\n";
								print_string "(* RÉSOLUTION COMPLÈTE *)\n";
								print_newline ();
								enregistrer_mouv cube.mouvement1.mv1 "mouv3333";
								let l = resoudre_le_cube cube true in
									dessine_cube cube;
									printf__printf "\n (* nombre de quarts de tours : %d *)\n" l;
									print_newline ()
			);
			((* mélanger *)
				fun () -> random__init (int_of_float (10000. *. sys__time ()));
								cube.mouvement1.mv1 <- mv1_rubik_r ();
								dessine_cube cube;
			);
			((* restituer *)
				fun () -> cube.mouvement1.mv1 <- lire_mouv ("mouv3333");
								dessine_cube cube;
			);
			((* composer *)
				fun () -> let hauteur = 500 and largeur = 900
								in
									let l = string_of_int largeur and h = string_of_int hauteur in
										graphics__close_graph ();
										boucle_saisie cube (" " ^ l ^ "x" ^ h);
										boucle cube
			);
		|]
	with graphics__Graphic_failure s -> ()
;;

let mouvement = {mv1 = lire_mouv "mouv3333"};;
let context = {matrice = id};;
let repere = {plan = (306, 306, 20, 20)};;
let anime = ref true;;
let cube = nouveau_cube mouvement context repere anime;;
boucle cube;;